<?php
/* --------------------------------------------------------------
 ShopStatusReader.php 2020-09-09
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

declare(strict_types=1);

namespace GXModules\GambioSamples\ShopStatusManager\App\Data;

use Doctrine\DBAL\Connection;
use Gambio\Core\Configuration\Services\ConfigurationFinder;
use InvalidArgumentException;

/**
 * Class ShopStatusReader
 *
 * @package GXModules\GambioSamples\ShopStatusManager\App\Data
 */
class ShopStatusReader
{
    private const TOP_BAR_TYPE = 'topbar';
    private const POPUP_TYPE   = 'popup';
    
    /**
     * @var Connection
     */
    private $connection;
    
    /**
     * @var ConfigurationFinder
     */
    private $finder;
    
    
    /**
     * @param Connection          $connection
     * @param ConfigurationFinder $finder
     */
    public function __construct(Connection $connection, ConfigurationFinder $finder)
    {
        $this->connection = $connection;
        $this->finder     = $finder;
    }
    
    
    /**
     * @return array
     */
    public function getPopupData(): array
    {
        $status = $this->stringToBool(
            $this->finder->get('gm_configuration/POPUP_NOTIFICATION_STATUS', '0')
        );
        
        return [
            'status'   => $status,
            'contents' => $this->contentData(self::POPUP_TYPE),
        ];
    }
    
    
    /**
     * @return array
     */
    public function getTopBarData(): array
    {
        $status = $this->stringToBool(
            $this->finder->get('gm_configuration/TOPBAR_NOTIFICATION_STATUS', '0')
        );
        $color  = $this->finder->get('gm_configuration/TOPBAR_NOTIFICATION_COLOR', '#ffffff');
        
        return [
            'status'   => $status,
            'color'    => $color,
            'contents' => $this->contentData(self::TOP_BAR_TYPE),
        ];
    }
    
    
    /**
     * @return array
     */
    public function getStatusData(): array
    {
        $status  = $this->stringToBool(
            $this->finder->get('gm_configuration/GM_SHOP_OFFLINE')
        );
        $message = $this->finder->get('gm_configuration/GM_SHOP_OFFLINE_MSG');
        
        return [
            'status'  => $status,
            'message' => $message
        ];
    }
    
    
    /**
     * @param string $value
     *
     * @return bool
     */
    private function stringToBool(string $value): bool
    {
        return $value === '1' || $value === 'true' || $value === 'checked';
    }
    
    
    /**
     * @param string $type
     *
     * @return array
     */
    private function contentData(string $type): array
    {
        $qb = $this->connection->createQueryBuilder();
        
        $selects = ['content', 'code'];
        $where   = "sn.notice_type = {$qb->createNamedParameter($type)}";
        
        $qb->select($selects)->from('shop_notices', 'sn')->leftJoin(
            'sn',
            'shop_notice_contents',
            'snc',
            'sn.shop_notice_id = snc.shop_notice_id'
        )->leftJoin('snc', 'languages', 'l', 'snc.language_id = l.languages_id')->where($where);
        
        return $qb->execute()->fetchAll();
    }
}