<?php
/* --------------------------------------------------------------
 ShopStatusWriter.php 2020-09-09
 Gambio GmbH
 http://www.gambio.de
 Copyright (c) 2020 Gambio GmbH
 Released under the GNU General Public License (Version 2)
 [http://www.gnu.org/licenses/gpl-2.0.html]
 --------------------------------------------------------------
 */

declare(strict_types=1);

namespace GXModules\GambioSamples\ShopStatusManager\App\Data;

use Doctrine\DBAL\Connection;
use Doctrine\DBAL\DBALException;
use Gambio\Core\Configuration\Services\ConfigurationService;
use GXModules\GambioSamples\ShopStatusManager\Model\Popup;
use GXModules\GambioSamples\ShopStatusManager\Model\OfflineStatus;
use GXModules\GambioSamples\ShopStatusManager\Model\Topbar;
use InvalidArgumentException;

/**
 * Class ShopStatusWriter
 *
 * @package GXModules\GambioSamples\ShopStatusManager\App\Data
 */
class ShopStatusWriter
{
    private const TOP_BAR_ID = 1;
    private const POPUP_ID   = 2;
    private const TABLE      = 'shop_notice_contents';
    
    /**
     * @var Connection
     */
    private $connection;
    
    /**
     * @var ConfigurationService
     */
    private $configurationService;
    
    
    /**
     * @param Connection           $connection
     * @param ConfigurationService $configurationService
     */
    public function __construct(Connection $connection, ConfigurationService $configurationService)
    {
        $this->connection           = $connection;
        $this->configurationService = $configurationService;
    }
    
    
    /**
     * @param Topbar $topbar
     *
     * @throws DBALException
     */
    public function saveTopbar(Topbar $topbar): void
    {
        $topbarData = $topbar->toArray();
        foreach ($topbarData['contents'] as $content) {
            $data       = ['content' => $content['content']];
            $identifier = [
                'shop_notice_id' => self::TOP_BAR_ID,
                'language_id'    => $this->languageCodeToId($content['languageCode']),
            ];
            
            $this->connection->update(self::TABLE, $data, $identifier);
        }
        
        $this->configurationService->save('gm_configuration/TOPBAR_NOTIFICATION_STATUS',
                                          $topbarData['status'] ? '1' : '0');
        $this->configurationService->save('gm_configuration/TOPBAR_NOTIFICATION_COLOR', $topbarData['color']);
    }
    
    
    /**
     * @param Popup $popup
     *
     * @throws DBALException
     */
    public function savePopup(Popup $popup): void
    {
        $popupData = $popup->toArray();
        foreach ($popupData['contents'] as $content) {
            $data       = ['content' => $content['content']];
            $identifier = [
                'shop_notice_id' => self::POPUP_ID,
                'language_id'    => $this->languageCodeToId($content['languageCode']),
            ];
            
            $this->connection->update(self::TABLE, $data, $identifier);
        }
        
        $this->configurationService->save('gm_configuration/POPUP_NOTIFICATION_STATUS',
                                          $popupData['status'] ? '1' : '0');
    }
    
    
    /**
     * @param OfflineStatus $shopStatus
     */
    public function saveStatus(OfflineStatus $shopStatus): void
    {
        $data = $shopStatus->toArray();
        
        $this->configurationService->save('gm_configuration/GM_SHOP_OFFLINE', $data['isOffline'] ? 'checked' : '');
        $this->configurationService->save('gm_configuration/GM_SHOP_OFFLINE_MSG', $data['message']);
    }
    
    
    /**
     * @param string $code
     *
     * @return string
     */
    private function languageCodeToId(string $code): string
    {
        $idColumn = 'languages_id';
        $qb       = $this->connection->createQueryBuilder();
        $where    = "code = {$qb->createNamedParameter($code)}";
        
        $qb->select($idColumn)->from('languages')->where($where);
        $result = $qb->execute()->fetch();
        
        if (empty($result)) {
            throw new InvalidArgumentException("Language (code: {$code}) not found.");
        }
        
        return $result[$idColumn];
    }
}